/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2015 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4, bitwise: true, node: true, expr: true*/
/*global describe: true, _: true, expect: true, window: true, WebSocket: true, it: true, afterEach, beforeEach */
"use strict";

var AssetExtractor = require("../lib/asset-extractor.js"),
    AssetExporter = require("../lib/asset-exporter"),
    FileUtils = require("../lib/file-utils"),
    Q = require("q"),
    path = require("path"),
    sinon = require("sinon");

describe('AssetExtractor', function() {
    var sandbox = sinon.sandbox.create(),
        connectionId = 21,
        layer = {
            id: 7,
            name: "batman"
        },
        layerComponent = {
            documentId: 3,
            layerId: 7,
            extension: "png",
            path: "/Users/Batman/Desktop/batman.png"
        },
        layerPreviewComponent = {
            timestamp: new Date().getTime(),
            documentId: 4,
            layerId: 8,
            extension: "jpg",
            scale: 0.5,
            quality: 60
        },
        document = {
            id: 3,
            basename: "gotham-skyline",
            layers: {
                findLayer: function(id) {
                    return id == layer.id ? { index: 1, layer: layer } : null;
                }
            }
        },
        documentComponent = {
            documentId: 3,
            extension: "png",
            path: "/Users/Batman/Desktop/gotham-skyline.png"
        },
        quickExportOptions = {
            fileType: "png",
            destFolder: "/Users/Batman/Desktop/"
        },
        fileSize = 567,
        fakeGenerateComponentToTempFile = function (component) {
            if (component.layerId) {
                component.layer = {};
            }
            return Q.resolve({ path: "/tmp/foo.png" });
        },
        componentBuffer = new Buffer([0,1,2,3,4,5,6,7]),
        fakeGenerateComponentToBuffer = function (component) {
            if (component.layerId) {
                component.layer = {};
            }
            return Q.resolve({});
        },
        fakeInvisibleComponent = function (component) {
            if (component.layerId) {
                component.layer = {
                    visible: false
                };
            }
            return Q.resolve({ path:"/tmp/foo.png" });
        };

    beforeEach(function () {
        sandbox.stub(AssetExtractor, "getDocument").returns(Q.resolve(document));
        AssetExtractor._logger = { log: sinon.stub(),
                                   error: sinon.stub() };
    });

    afterEach(function () {
        AssetExtractor._logger = null;
        sandbox.restore();
    });

    it("should export a layer component", function () {
        // Pretend AssetExporter generates a temporary asset file when called.
        sandbox.stub(AssetExporter, "exportAsset").returns(Q.resolve({ path: "/tmp/123" }));

        // Observe the file operation to verify behavior.
        sandbox.stub(FileUtils, "moveTemporaryFile").returns(Q.resolve());

        var promise = AssetExtractor.exportComponent(layerComponent);
        return expect(promise).to.eventually.be.fulfilled
            .then(function () {
                expect(FileUtils.moveTemporaryFile).to.have.been.calledOnce;
                expect(FileUtils.moveTemporaryFile).to.have.been.calledWith("/tmp/123", "/Users/Batman/Desktop/batman.png");
            });
    });

    it("should export a document component", function () {
        // Pretend AssetExporter generates a temporary asset file when called.
        sandbox.stub(AssetExporter, "exportAsset").returns(Q.resolve({ path: "/tmp/456" }));

        // Observe the file operation to verify behavior.
        sandbox.stub(FileUtils, "moveTemporaryFile").returns(Q.resolve());

        var promise = AssetExtractor.exportComponent(documentComponent);
        return expect(promise).to.eventually.be.fulfilled
            .then(function () {
                expect(FileUtils.moveTemporaryFile).to.have.been.calledOnce;
                expect(FileUtils.moveTemporaryFile).to.have.been.calledWith("/tmp/456", "/Users/Batman/Desktop/gotham-skyline.png");
            });
    });

    it("should export multiple components", function () {
        sandbox.stub(AssetExtractor, "exportComponent").returns(Q.resolve());
        sandbox.stub(FileUtils, "openFolderOnceInOS");
        sandbox.stub(path, "dirname").returns("hello world");

        var components = [layerComponent, documentComponent],
            promise = AssetExtractor.exportComponents(components);

        return expect(promise).to.eventually.be.fulfilled
            .then(function () {
                expect(AssetExtractor.exportComponent).to.have.been.calledTwice;
                expect(AssetExtractor.exportComponent).to.have.been.calledWith(layerComponent);
                expect(AssetExtractor.exportComponent).to.have.been.calledWith(documentComponent);
                expect(FileUtils.openFolderOnceInOS).to.have.been.calledWith("hello world");
            });
    });

    it("should not open the finder window if no components were successfully exported", function () {
        // Pretend none of the assets could be written to disk due to permissions errors.
        sandbox.stub(AssetExtractor, "exportComponent").returns(Q.reject(new FileUtils.PermissionsWriteError()));
        sandbox.stub(FileUtils, "openFolderOnceInOS");
        sandbox.stub(path, "dirname").returns("hello world");

        var components = [layerComponent, documentComponent],
            promise = AssetExtractor.exportComponents(components);

        return expect(promise).to.eventually.be.fulfilled
            .then(function () {
                expect(AssetExtractor.exportComponent).to.have.been.calledTwice;
                expect(AssetExtractor.exportComponent).to.have.been.calledWith(layerComponent);
                expect(AssetExtractor.exportComponent).to.have.been.calledWith(documentComponent);
                expect(FileUtils.openFolderOnceInOS).to.not.have.been.called;
            });
    });

    it("should generate an item from the asset exporter", function () {
        // Pretend AssetExporter generates a temporary asset file when called.
        sandbox.stub(AssetExporter, "exportAsset").returns(Q.resolve({ path: "/tmp/123" }));

        var promise = AssetExtractor.generateComponent(layerComponent);
        return expect(promise).to.eventually.have.property("path", "/tmp/123");
    });
    
    it("should get the file size and delete the tmp component", function () {
        sandbox.stub(AssetExtractor, "generateComponent", fakeGenerateComponentToTempFile);
        sandbox.stub(FileUtils, "stat").returns(Q.resolve({size: fileSize}));
        sandbox.stub(FileUtils, "removeFile").returns(Q.resolve());

        var component = {name: "hello world"},
            promise = AssetExtractor.getComponentFileSize(component);
        return expect(promise).to.eventually.be.fulfilled
            .then(function (result) {
                expect(AssetExtractor.generateComponent).to.have.been.calledOnce;
                expect(AssetExtractor.generateComponent).to.have.been.calledWith(component);
                expect(FileUtils.stat).to.have.been.calledOnce;
                expect(FileUtils.stat).to.have.been.calledWith("/tmp/foo.png");
                expect(FileUtils.removeFile).to.have.been.calledOnce;
                expect(FileUtils.removeFile).to.have.been.calledWith("/tmp/foo.png");
                expect(result).to.equal(fileSize);
            });
    });

    describe("previews", function () {

        beforeEach(function () {
            sandbox.stub(FileUtils, "stat").returns(Q.resolve({ size: fileSize }));
        });

        it("should generate a preview to temp file and create a component rendered event", function () {
            sandbox.stub(AssetExtractor, "generateComponent", fakeGenerateComponentToTempFile);
            return expect(AssetExtractor.generatePreview(layerPreviewComponent, connectionId)).to.eventually.be.fulfilled
                .then(function (result) {
                    expect(result).to.eql({
                        componentId: 1,
                        documentId: layerPreviewComponent.documentId,
                        errors: null,
                        eventType: "componentRendered",
                        fileSize: fileSize,
                        hasZeroBounds: false,
                        invisible: false,
                        layerId: layerPreviewComponent.layerId,
                        outsideDocumentBounds: false,
                        scale: layerPreviewComponent.scale,
                        timestamp: layerPreviewComponent.timestamp
                    });
                });
        });

        it("should generate a preview to buffer and create a component rendered event", function () {
            sandbox.stub(AssetExtractor, "generateComponent", fakeGenerateComponentToBuffer);
            return expect(AssetExtractor.generatePreview(layerPreviewComponent, connectionId)).to.eventually.be.fulfilled
                .then(function (result) {
                    expect(result).to.eql({
                        componentId: 2,
                        documentId: layerPreviewComponent.documentId,
                        errors: null,
                        eventType: "componentRendered",
                        fileSize: 0,
                        hasZeroBounds: false,
                        invisible: false,
                        layerId: layerPreviewComponent.layerId,
                        outsideDocumentBounds: false,
                        scale: layerPreviewComponent.scale,
                        timestamp: layerPreviewComponent.timestamp
                    });
                });
        });

        it("should include group visiblity information in the component rendered event", function () {
            sandbox.stub(AssetExtractor, "generateComponent", fakeGenerateComponentToTempFile);
            return expect(AssetExtractor.generatePreview(layerPreviewComponent, connectionId)).to.eventually.be.fulfilled
                .then(function (result) {
                    expect(result.invisible).to.equal(false);
                });
        });

        it("should consider hidden layers as visible for extracting", function () {
            sandbox.stub(AssetExtractor, "generateComponent", fakeInvisibleComponent);
            return expect(AssetExtractor.generatePreview(layerPreviewComponent, connectionId)).to.eventually.be.fulfilled
                .then(function (result) {
                    expect(result.invisible).to.equal(false);
                });
        });

        it("should include info about zero bounds layers in the component rendered event", function () {
            var errorMessage = "Layer bounds are zero.";
            sandbox.stub(AssetExtractor, "generateComponent").returns(Q.reject({
                zeroBoundsError: true,
                message: errorMessage
            }));
            return expect(AssetExtractor.generatePreview(layerPreviewComponent, connectionId)).to.eventually.be.fulfilled
                .then(function (result) {
                    expect(result.hasZeroBounds).to.equal(true);
                    expect(result.errors).to.eql([errorMessage]);
                });
        });

        it("should include info about layers that are out of doc bounds in the component rendered event", function () {
            var errorMessage = "Layer is out of doc bounds.";
            sandbox.stub(AssetExtractor, "generateComponent").returns(Q.reject({
                outsideDocumentBoundsError: true,
                message: errorMessage
            }));
            return expect(AssetExtractor.generatePreview(layerPreviewComponent, connectionId)).to.eventually.be.fulfilled
                .then(function (result) {
                    expect(result.outsideDocumentBounds).to.equal(true);
                    expect(result.errors).to.eql([errorMessage]);
                });
        });

        it("should include arbitrary error or warning messages from the rendering pipeline", function () {
            var errorMessage = "Something isn't quite right!";
            sandbox.stub(AssetExtractor, "generateComponent").returns(Q.reject({ message: errorMessage }));
            return expect(AssetExtractor.generatePreview(layerPreviewComponent, connectionId)).to.eventually.be.fulfilled
                .then(function (result) {
                    expect(result.hasZeroBounds).to.equal(false);
                    expect(result.outsideDocumentBounds).to.equal(false);
                    expect(result.errors).to.eql([errorMessage]);
                });
        });

        it("should include arbitrary error or warning messages from the rendering pipeline on success", function () {
            var errorMessageArray = ["Something isn't quite right!"];
            sandbox.stub(AssetExtractor, "generateComponent").returns(Q.resolve({ errors: errorMessageArray }));
            return expect(AssetExtractor.generatePreview(layerPreviewComponent, connectionId)).to.eventually.be.fulfilled
                .then(function (result) {
                    expect(result.errors).to.eql(errorMessageArray);
                });
        });

        it("should retain the temporary file path of a previously generated preview", function () {
            sandbox.stub(AssetExtractor, "generateComponent", fakeGenerateComponentToTempFile);
            return expect(AssetExtractor.generatePreview(layerPreviewComponent, connectionId)).to.eventually.be.fulfilled
                .then(function (result) {
                    var previewId = result.componentId;
                    expect(AssetExtractor.getPreview(previewId).tempAssetPath).to.equal("/tmp/foo.png");
                });
        });

        it("should not generate a temporary file path of a generated preview to buffer", function () {
            sandbox.stub(AssetExtractor, "generateComponent", fakeGenerateComponentToBuffer);
            return expect(AssetExtractor.generatePreview(layerPreviewComponent, connectionId)).to.eventually.be.fulfilled
                .then(function (result) {
                    var previewId = result.componentId;
                    expect(AssetExtractor.getPreview(previewId).path).to.be.equal(undefined);
                });
        });

        it("should not return a preview for an asset that was not generated", function () {
            var previewId = 123; // Bogus id.
            expect(AssetExtractor.getPreview(previewId)).to.equal(undefined);
        });

        it("should delete preview temp files associated with a connection", function () {
            sandbox.stub(AssetExtractor, "generateComponent", fakeGenerateComponentToTempFile);
            return expect(AssetExtractor.generatePreview(layerPreviewComponent, connectionId)).to.eventually.be.fulfilled
                .then(function (result) {
                    var previewId = result.componentId;
                    expect(AssetExtractor.getPreview(previewId).tempAssetPath).to.equal("/tmp/foo.png");

                    AssetExtractor.deleteAllPreviewsForConnection(connectionId);
                    expect(AssetExtractor.getPreview(previewId)).to.equal(undefined);
                });
        });

        it("should recognize layer with visible:true as visible", function () {
            var layer = {
                visible: true
            };
            expect(AssetExtractor._isVisible(layer)).to.equal(true);
        });

        it("should recognize layer with visible:false as invisible", function () {
            var layer = {
                visible: false
            };
            expect(AssetExtractor._isVisible(layer)).to.equal(false);
        });

        it("should recognize layer with visible undefined as visible", function () {
            var layer = {};
            expect(AssetExtractor._isVisible(layer)).to.equal(true);
        });

        it("should recognize layer with at least 1 visible sublayer as visible", function () {
            var layer = {
                visible: true,
                layers: [
                    { visible: false },
                    { visible: false },
                    { visible: true }
                ]
            };
            expect(AssetExtractor._isVisible(layer)).to.equal(true);
        });

        it("should recognize layer with no visible sublayers as invisible", function () {
            var layer = {
                visible: true,
                layers: [
                    { visible: false },
                    { visible: false },
                    { visible: false }
                ]
            };
            expect(AssetExtractor._isVisible(layer)).to.equal(false);
        });
    });
});
